<?php
// Drop-in replacement: White theme + improved charts (donut + gauge)
// This file merges your latest layout with: forced white theme, refined donut with labels + tooltip, polished gauge.

// ==================== PHP: Prepare Data ====================
$BASE        = $BASE ?? '';
$allProjects = $seed['projects'] ?? [];
$allJobs     = $seed['jobs'] ?? [];

if (!function_exists('can_view_all')) { function can_view_all(){ return true; } }
if (!function_exists('user_team'))    { function user_team(){ return ''; } }

if (!can_view_all()) {
  $team = user_team();
  $allProjects = array_values(array_filter($allProjects, fn($p)=> ($p['team']??'') === $team));
  $allowedIds  = array_column($allProjects, 'id');
  $allJobs     = array_values(array_filter($allJobs, fn($j)=> in_array($j['project_id'], $allowedIds, true)));
}

function bucket_status($raw) {
  $s = strtolower(trim((string)$raw));
  if (in_array($s, ['done','complete','completed','finished','closed'])) return 'completed';
  if (in_array($s, ['in progress','active','ongoing','running','executing'])) return 'in_progress';
  if (in_array($s, ['planned','planning','scheduled'])) return 'planned';
  if (in_array($s, ['on hold','paused','hold','waiting'])) return 'on_hold';
  return 'other';
}
function safe_lc($s){ $s = (string)($s ?? ''); return strtolower($s); }

// index jobs by project
$jobsByProject = [];
foreach ($allJobs as $j) {
  $pid = (int)($j['project_id'] ?? 0);
  if ($pid > 0) $jobsByProject[$pid][] = $j;
}

function project_percent(array $p, array $jobsByProject): int {
  if (isset($p['progress'])) { $x = (int)$p['progress']; return max(0, min(100, $x)); }
  $pid   = (int)($p['id'] ?? 0);
  $items = $jobsByProject[$pid] ?? [];
  if (!$items) return 0;
  $total = 0.0; $done = 0.0;
  foreach ($items as $t) {
    $w = isset($t['plan_hours']) ? max(1.0, (float)$t['plan_hours']) : 1.0;
    $total += $w;
    $status = strtolower((string)($t['status'] ?? ''));
    if (isset($t['progress'])) $tp = max(0.0, min(100.0, (float)$t['progress']));
    else $tp = ($status === 'done' || $status === 'completed') ? 100.0 : (($status === 'active') ? 50.0 : (($status === 'planning') ? 10.0 : 0.0));
    $done += $w * ($tp / 100.0);
  }
  return (int)round(100.0 * $done / max(1.0, $total));
}

// compute buckets + percents
$counters = ['completed'=>0,'in_progress'=>0,'planned'=>0,'on_hold'=>0,'other'=>0];
$percents = [];
foreach ($allProjects as &$p) {
  $p['_bucket'] = bucket_status($p['status'] ?? '');
  $counters[$p['_bucket']]++;
  $percents[(int)$p['id']] = project_percent($p, $jobsByProject);
}
unset($p);
$total = count($allProjects);

$chartData = [
  'Completed'   => $counters['completed'],
  'In Progress' => $counters['in_progress'],
  'Planned'     => $counters['planned'],
  'On Hold'     => $counters['on_hold'],
  'Other'       => $counters['other'],
];

$scoreBase    = max(1, $total);
$overallScore = (int)round((($counters['completed'] + $counters['in_progress'] + $counters['planned']) / $scoreBase) * 100 * 0.88);

$teams = array_values(array_unique(array_map(fn($p)=> (string)($p['team'] ?? ''), $seed['projects'] ?? [])));
sort($teams, SORT_NATURAL|SORT_FLAG_CASE);

$bucketName = [ 'completed'=>'Completed','in_progress'=>'In Progress','planned'=>'Planned','on_hold'=>'On Hold','other'=>'Other' ];
?>

<section class="shell">
  <!-- ===== Sidebar (filters) ===== -->
  <aside class="side">
    <div class="brand">
      <div class="logo">📊</div>
      <div class="btx">
        <b>Portfolio</b>
        <small>Project Dashboard</small>
      </div>
    </div>

    <div class="box">
      <div class="box-h">Search</div>
      <div class="search">
        <input id="q" type="search" placeholder="Search name/code/team" aria-label="Search projects">
        <button id="clearQ" class="btn tiny" title="Clear">✕</button>
      </div>
    </div>

    <div class="box">
      <div class="box-h">Filters</div>
      <label class="lbl">Team
        <select id="teamFilter" aria-label="Filter by team">
          <option value="">All teams</option>
          <?php foreach ($teams as $t): if ($t!==''): ?>
            <option value="<?= htmlspecialchars($t) ?>"><?= htmlspecialchars($t) ?></option>
          <?php endif; endforeach; ?>
        </select>
      </label>
      <label class="lbl">Status
        <select id="statusFilter" aria-label="Filter by status">
          <option value="all">All</option>
          <option value="completed">Completed</option>
          <option value="in_progress">In Progress</option>
          <option value="planned">Planned</option>
          <option value="on_hold">On Hold</option>
          <option value="other">Other</option>
        </select>
      </label>
      <label class="lbl">Sort by
        <select id="sorter" aria-label="Sort projects">
          <option value="name_asc">Name ↑</option>
          <option value="name_desc">Name ↓</option>
          <option value="status">Status</option>
          <option value="team">Team</option>
          <option value="progress_desc">% Progress ↓</option>
          <option value="progress_asc">% Progress ↑</option>
          <option value="pinned_first">Pinned first</option>
        </select>
      </label>
      <div class="grp">
        <button id="density" class="btn w100" title="Toggle density">Density</button>
        <button id="exportCsv" class="btn w100" title="Export visible as CSV">Export CSV</button>
      </div>
    </div>

    <div class="box foot">
      <div class="tips">Tip: คุณสามารถปักหมุด (★) โปรเจกต์สำคัญและเลือกมุมมองบอร์ดเพื่อโฟกัสที่สถานะได้</div>
    </div>
  </aside>

  <!-- ===== Main ===== -->
  <main class="main">
    <header class="top">
      <div>
        <h1>Project Portfolio</h1>
        <p class="sub">ภาพรวมโครงการทั้งหมด (<?= (int)$total ?>)</p>
      </div>
      <div class="actions">
        <div class="seg" role="group" aria-label="View switch">
          <button class="seg-btn active" data-view="board" aria-pressed="true">Board</button>
          <button class="seg-btn" data-view="table" aria-pressed="false">Table</button>
        </div>
        <button id="fullscreen" class="btn">Fullscreen</button>
        <button id="refresh" class="btn">Refresh</button>
        <?php if (function_exists('can_plan') && can_plan()): ?>
          <a class="btn primary" href="<?= $BASE ?>/projects">+ New Project</a>
        <?php endif; ?>
      </div>
    </header>

    <!-- KPIs -->
    <div class="kpis">
      <div class="kpi"><div class="k">Total</div><div class="v"><?= (int)$total ?></div></div>
      <div class="kpi ok"><div class="k">Completed</div><div class="v"><?= (int)$counters['completed'] ?></div></div>
      <div class="kpi info"><div class="k">In Progress</div><div class="v"><?= (int)$counters['in_progress'] ?></div></div>
      <div class="kpi warn"><div class="k">Planned</div><div class="v"><?= (int)$counters['planned'] ?></div></div>
      <div class="kpi hold"><div class="k">On Hold</div><div class="v"><?= (int)$counters['on_hold'] ?></div></div>
      <div class="kpi vio"><div class="k">Other</div><div class="v"><?= (int)$counters['other'] ?></div></div>
    </div>

    <!-- Charts -->
    <div class="grid2">
      <div class="card chart">
        <div class="h">Project Status Breakdown</div>
        <div id="donut" class="donut" role="img" aria-label="Status share"></div>
        <div class="legend">
          <?php foreach ($chartData as $label => $val): ?>
            <div class="lgd"><span class="dot" data-label="<?= htmlspecialchars($label) ?>"></span><?= htmlspecialchars($label) ?> <b><?= (int)$val ?></b></div>
          <?php endforeach; ?>
        </div>
      </div>
      <div class="card health">
        <div class="h">Portfolio Health</div>
        <div class="gauge" data-score="<?= (int)$overallScore ?>">
          <svg viewBox="0 0 220 140" class="gsvg" aria-hidden="true">
            <!-- Track -->
            <path d="M20 120 A90 90 0 0 1 200 120" class="trk"/>
            <!-- Fill (uses stroke-dasharray) -->
            <path id="gFill" d="M20 120 A90 90 0 0 1 200 120" class="fill"/>
            <!-- Ticks -->
            <?php for($i=0;$i<=10;$i++): $x=20 + (180*($i/10)); ?>
              <line x1="<?= 20 + ($i*18) ?>" y1="120" x2="<?= 20 + ($i*18) ?>" y2="116" class="tick" />
            <?php endfor; ?>
          </svg>
          <div class="n"><?= (int)$overallScore ?><span>%</span></div>
          <div class="s">Overall Score</div>
        </div>
        <div class="meta">
          <span class="pill ok">Passing: <b><?= ($counters['completed']+$counters['in_progress']+$counters['planned']) ?></b></span>
          <span class="pill warn">Warnings: <b><?= $counters['on_hold'] ?></b></span>
          <span class="pill crit">Other: <b><?= $counters['other'] ?></b></span>
        </div>
      </div>
    </div>

    <!-- Board (Kanban) -->
    <section id="board" class="board" aria-live="polite">
      <?php foreach (['planned','in_progress','on_hold','completed','other'] as $col): ?>
        <div class="col" data-col="<?= $col ?>">
          <div class="col-h">
            <span class="badge <?= $col ?>"></span>
            <b><?= $bucketName[$col] ?></b>
            <small class="count" data-count="<?= $col ?>">0</small>
          </div>
          <div class="col-body"></div>
        </div>
      <?php endforeach; ?>
    </section>

    <!-- Table -->
    <section id="tableWrap" class="tablewrap" hidden>
      <table class="tbl" id="projectsTable">
        <thead><tr>
          <th style="width:1%"></th>
          <th>Project</th><th>Code</th><th>Team</th><th>Status</th><th>%</th><th>Action</th>
        </tr></thead>
        <tbody id="projectsTableBody">
          <?php foreach ($allProjects as $p): $id=(int)$p['id']; $pct=$percents[$id] ?? 0; ?>
            <tr
              data-id="<?= $id ?>"
              data-bucket="<?= htmlspecialchars($p['_bucket']) ?>"
              data-name="<?= htmlspecialchars(safe_lc($p['name'])) ?>"
              data-code="<?= htmlspecialchars(safe_lc($p['code'])) ?>"
              data-team="<?= htmlspecialchars(safe_lc($p['team'])) ?>"
              data-display-name="<?= htmlspecialchars($p['name']) ?>"
              data-display-code="<?= htmlspecialchars($p['code']) ?>"
              data-display-team="<?= htmlspecialchars($p['team']) ?>"
              data-display-status="<?= htmlspecialchars($p['status']) ?>"
              data-progress="<?= $pct ?>"
              data-pinned="false"
            >
              <td class="pin-cell"><button class="pin" type="button" aria-label="Pin">☆</button></td>
              <td class="t-name"><?= htmlspecialchars($p['name']) ?></td>
              <td><?= htmlspecialchars($p['code']) ?></td>
              <td><?= htmlspecialchars($p['team']) ?></td>
              <td><span class="st <?= htmlspecialchars($p['_bucket']) ?>"><?= htmlspecialchars($p['status']) ?></span></td>
              <td class="t-pct">
                <div class="prog tiny"><div class="bar" style="--pct: <?= $pct ?>%"></div></div>
                <span class="pct"><?= $pct ?>%</span>
              </td>
              <td><button class="link small" data-open-detail>View</button></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </section>

    <div id="emptyState" class="empty" hidden>
      <div class="emoji">🔎</div>
      <div class="t1">No projects match your filters</div>
      <div class="t2">Clear search or change filters</div>
    </div>
  </main>

  <!-- Drawer -->
  <aside id="drawer" class="drawer" aria-hidden="true">
    <div class="panel">
      <header class="dh">
        <div>
          <div class="ttl" id="dName">—</div>
          <div class="sub" id="drawerSub"></div>
        </div>
        <button class="btn ghost" id="drawerClose" aria-label="Close">✕</button>
      </header>
      <div class="db">
        <div class="kv"><span>Code</span><b id="dCode">—</b></div>
        <div class="kv"><span>Team</span><b id="dTeam">—</b></div>
        <div class="kv"><span>Status</span><span class="st" id="dStatus">—</span></div>
        <div class="kv"><span>Progress</span>
          <div style="min-width:160px">
            <div class="prog tiny"><div class="bar" id="dProgBar" style="--pct:0%"></div></div>
            <div class="progmeta"><span class="pct" id="dProgPct">0%</span></div>
          </div>
        </div>
        <div class="div"></div>
        <div class="btitle">Suggested next steps</div>
        <ul class="bullets">
          <li>ตรวจสอบ milestone ล่าสุด & ผู้รับผิดชอบ</li>
          <li>อัปเดตสถานะและความเสี่ยง</li>
          <li>แนบเอกสารหรือ Link ที่เกี่ยวข้อง</li>
        </ul>
      </div>
      <footer class="df">
        <a id="dOpen" class="btn primary" href="#" target="_self">Open project</a>
        <button class="btn" id="drawerClose2">Close</button>
      </footer>
    </div>
  </aside>
</section>

<script>
(function(){
  const $  = (s, r=document)=> r.querySelector(s);
  const $$ = (s, r=document)=> Array.from(r.querySelectorAll(s));
  const LS = { get(k,d=null){ try{return JSON.parse(localStorage.getItem(k))??d}catch{return d} }, set(k,v){ localStorage.setItem(k, JSON.stringify(v)); } };

  // ===== Donut (improved) =====
  const data   = <?php echo json_encode(array_values($chartData), JSON_UNESCAPED_UNICODE); ?>;
  const labels = <?php echo json_encode(array_keys($chartData),   JSON_UNESCAPED_UNICODE); ?>;
  const total  = data.reduce((a,b)=>a+b,0) || 1;
  const donut  = $('#donut');
  const cls = (label)=>{ const t=(label||'').toLowerCase(); if(t.includes('complete'))return'completed'; if(t.includes('progress'))return'in_progress'; if(t.includes('planned'))return'planned'; if(t.includes('hold'))return'on_hold'; return'other'; };

  // Tooltip
  const tip = document.createElement('div'); tip.className='tip hidden'; donut.appendChild(tip);

  function arc(cx,cy,r,start,end){
    const s = polar(cx,cy,r,end); // draw large to small using stroke-dasharray; path is full circle
    return `M ${cx} ${cy} m -${r},0 a ${r},${r} 0 1,0 ${r*2},0 a ${r},${r} 0 1,0 -${r*2},0`;
  }
  function polar(cx,cy,r,a){ return { x: cx + r*Math.cos(a), y: cy + r*Math.sin(a) }; }

  function renderDonut(){
    const size = Math.min(donut.clientWidth||260, 360);
    const cx=size/2, cy=size/2, r=Math.max(56, Math.floor(size*0.32));
    const sw=Math.max(16, Math.floor(size*0.12));
    const C = 2*Math.PI*r;
    const svg = document.createElementNS('http://www.w3.org/2000/svg','svg'); svg.setAttribute('viewBox', `0 0 ${size} ${size}`); svg.classList.add('dsvg');

    // base ring
    const base = document.createElementNS('http://www.w3.org/2000/svg','circle');
    base.setAttribute('cx', cx); base.setAttribute('cy', cy); base.setAttribute('r', r);
    base.setAttribute('fill','none'); base.setAttribute('stroke','var(--line)'); base.setAttribute('stroke-width', sw);
    base.setAttribute('stroke-linecap','round'); svg.appendChild(base);

    // segments
    let offset=0; data.forEach((val,i)=>{
      if(!val) return;
      const len=C*(val/total);
      const c=document.createElementNS('http://www.w3.org/2000/svg','circle');
      c.setAttribute('cx', cx); c.setAttribute('cy', cy); c.setAttribute('r', r);
      c.setAttribute('fill','none'); c.setAttribute('stroke-width', sw);
      c.setAttribute('class','ring '+cls(labels[i]));
      c.setAttribute('stroke-dasharray', `${len} ${C-len}`);
      c.setAttribute('stroke-dashoffset', String(-(offset)));
      c.setAttribute('transform', `rotate(-90 ${cx} ${cy})`);
      c.setAttribute('stroke-linecap','round');
      c.addEventListener('mousemove', (e)=>{
        tip.textContent = `${labels[i]}: ${val}`; tip.classList.remove('hidden'); tip.style.left = (e.offsetX+10)+'px'; tip.style.top=(e.offsetY+10)+'px';
        centerTop.textContent = labels[i]; centerBot.textContent = val + ' / ' + total;
      });
      c.addEventListener('mouseleave', ()=>{ tip.classList.add('hidden'); centerTop.textContent='Total'; centerBot.textContent=total; });
      svg.appendChild(c); offset+=len;
    });

    // hole + center text
    const hole = document.createElementNS('http://www.w3.org/2000/svg','circle');
    hole.setAttribute('cx', cx); hole.setAttribute('cy', cy); hole.setAttribute('r', Math.max(44, r*0.62)); hole.setAttribute('class','donut-hole'); svg.appendChild(hole);

    const centerTop = document.createElementNS('http://www.w3.org/2000/svg','text');
    centerTop.setAttribute('x', cx); centerTop.setAttribute('y', cy-4); centerTop.setAttribute('text-anchor','middle'); centerTop.setAttribute('class','ctop'); centerTop.textContent='Total';
    const centerBot = document.createElementNS('http://www.w3.org/2000/svg','text');
    centerBot.setAttribute('x', cx); centerBot.setAttribute('y', cy+18); centerBot.setAttribute('text-anchor','middle'); centerBot.setAttribute('class','cbot'); centerBot.textContent=total;
    svg.appendChild(centerTop); svg.appendChild(centerBot);

    donut.innerHTML=''; donut.appendChild(svg);
    $$('.legend .dot').forEach(el=> el.classList.add(cls(el.getAttribute('data-label')||'')));
  }
  renderDonut(); if('ResizeObserver' in window){ new ResizeObserver(renderDonut).observe(donut); }

  // ===== Elements =====
  const board = $('#board');
  const tableW= $('#tableWrap');
  const tbody = $('#projectsTableBody');
  const qInp  = $('#q');
  const teamSel = $('#teamFilter');
  const statusSel = $('#statusFilter');
  const sortSel = $('#sorter');

  // Build payloads from table rows
  const rows = $$('#projectsTableBody tr');
  const payloads = rows.map(tr=> ({
    id: tr.dataset.id,
    bucket: tr.dataset.bucket,
    name: tr.dataset.displayName,
    code: tr.dataset.displayCode,
    team: tr.dataset.displayTeam,
    statusText: tr.dataset.displayStatus,
    progress: parseInt(tr.dataset.progress||'0',10),
    href: '<?= $BASE ?>/project?id=' + tr.dataset.id,
    pinned: tr.dataset.pinned==='true'
  }));

  // Helpers
  function debounce(fn, ms=200){ let t; return (...args)=>{ clearTimeout(t); t=setTimeout(()=>fn(...args), ms); } }
  const cmp = {
    name_asc:(a,b)=> a.name.localeCompare(b.name),
    name_desc:(a,b)=> b.name.localeCompare(a.name),
    status:(a,b)=> (a.bucket||'').localeCompare(b.bucket||'') || a.name.localeCompare(b.name),
    team:(a,b)=> (a.team||'').localeCompare(b.team||'') || a.name.localeCompare(b.name),
    progress_desc:(a,b)=> (b.progress-a.progress) || a.name.localeCompare(b.name),
    progress_asc:(a,b)=> (a.progress-b.progress) || a.name.localeCompare(b.name),
    pinned_first:(a,b)=> ((b.pinned?1:0)-(a.pinned?1:0)) || a.name.localeCompare(b.name),
  };

  // Render a card for board
  function cardHTML(p){
    return `<a class="card project" href="${p.href}" data-id="${p.id}" data-bucket="${p.bucket}" data-progress="${p.progress}" data-pinned="${p.pinned}">
      <div class="ph">
        <div class="pt" title="${escapeHtml(p.name)}">${escapeHtml(p.name)}</div>
        <span class="st ${p.bucket}">${escapeHtml(p.statusText)}</span>
      </div>
      <div class="pm">
        <div>Code: <b>${escapeHtml(p.code)}</b></div>
        <div>Team: ${escapeHtml(p.team)}</div>
      </div>
      <div class="prog"><div class="bar" style="--pct:${p.progress}%"></div></div>
      <div class="pm2"><span class="pct">${p.progress}%</span><button class="pin" type="button" aria-label="Pin">${p.pinned?'★':'☆'}</button></div>
      <div class="pf"><button class="link see" type="button" data-open-detail>Details</button></div>
    </a>`;
  }
  function escapeHtml(s){ return (s??'').toString().replaceAll('&','&amp;').replaceAll('<','&lt;').replaceAll('>','&gt;').replaceAll('"','&quot;'); }

  // State
  let currentView = 'board';

  function visibleFilter(p){
    const q = (qInp.value||'').trim().toLowerCase();
    const team = (teamSel.value||'').toLowerCase();
    const st = (statusSel.value||'all');
    const qok = !q || p.name.toLowerCase().includes(q) || p.code.toLowerCase().includes(q) || (p.team||'').toLowerCase().includes(q);
    const tok = !team || (p.team||'').toLowerCase()===team;
    const sok = (st==='all' || p.bucket===st);
    return qok && tok && sok;
  }

  function renderBoard(){
    const list = payloads.filter(visibleFilter).slice();
    list.sort(cmp[sortSel.value]||cmp.name_asc);
    $$('#board .col .col-body').forEach(el=> el.innerHTML='');
    const count = {planned:0,in_progress:0,on_hold:0,completed:0,other:0};
    list.forEach(p=>{ const c = document.querySelector(`#board .col[data-col="${p.bucket}"] .col-body`); if(c){ c.insertAdjacentHTML('beforeend', cardHTML(p)); count[p.bucket]++; }});
    Object.keys(count).forEach(k=>{ const t = document.querySelector(`.count[data-count="${k}"]`); if(t) t.textContent = count[k]; });
    bindCardEvents();
    document.querySelector('#emptyState').hidden = list.length>0;
  }

  function renderTable(){
    const list = payloads.filter(visibleFilter).slice();
    list.sort(cmp[sortSel.value]||cmp.name_asc);
    const map = new Map(list.map(p=>[String(p.id),true]));
    let visible=0;
    $$('#projectsTableBody tr').forEach(tr=>{ const show = map.has(tr.dataset.id); tr.style.display = show?'':'none'; if(show) visible++; });
    document.querySelector('#emptyState').hidden = visible>0;
  }

  const onChange = debounce(()=>{ currentView==='board' ? renderBoard() : renderTable(); LS.set('ppt_q', qInp.value); LS.set('ppt_team', teamSel.value); LS.set('ppt_status', statusSel.value); }, 150);
  qInp.addEventListener('input', onChange);
  teamSel.addEventListener('change', onChange);
  statusSel.addEventListener('change', onChange);
  sortSel.addEventListener('change', ()=>{ LS.set('ppt_sort', sortSel.value); currentView==='board' ? renderBoard() : renderTable(); });
  document.getElementById('clearQ').addEventListener('click', ()=>{ qInp.value=''; qInp.focus(); onChange(); });

  // Density toggle
  document.getElementById('density').addEventListener('click', (e)=>{
    const cur = document.getElementById('board').getAttribute('data-density')||'normal';
    const next = (cur==='normal')? 'compact':'normal';
    document.getElementById('board').setAttribute('data-density', next);
    e.currentTarget.setAttribute('aria-pressed', String(next==='compact'));
    LS.set('ppt_density', next);
  });

  // View switch
  $$('.seg-btn').forEach(b=> b.addEventListener('click', ()=>{
    currentView = b.dataset.view;
    $$('.seg-btn').forEach(x=>{ x.classList.toggle('active', x===b); x.setAttribute('aria-pressed', x===b); });
    if(currentView==='board'){ document.getElementById('board').hidden=false; document.getElementById('tableWrap').hidden=true; renderBoard(); }
    else { document.getElementById('board').hidden=true; document.getElementById('tableWrap').hidden=false; renderTable(); }
    LS.set('ppt_view', currentView);
  }));

  // Fullscreen/Refresh
  document.getElementById('fullscreen').addEventListener('click', ()=>{ const el=document.documentElement; !document.fullscreenElement ? el.requestFullscreen?.() : document.exitFullscreen?.(); });
  document.getElementById('refresh').addEventListener('click', ()=> location.reload());

  // Export CSV
  document.getElementById('exportCsv').addEventListener('click', ()=>{
    const list = payloads.filter(visibleFilter).slice();
    list.sort(cmp[sortSel.value]||cmp.name_asc);
    const headers=['id','name','code','team','status','progress','pinned'];
    const csv=[headers.join(',')].concat(list.map(p=> headers.map(h=>{ const v=(h==='status')?p.statusText:(p[h]??''); return '"'+String(v).replaceAll('"','""')+'"'; }).join(','))).join('\r\n');
    const blob=new Blob([csv],{type:'text/csv;charset=utf-8;'}); const a=document.createElement('a'); a.href=URL.createObjectURL(blob); a.download='projects_visible.csv'; a.click(); URL.revokeObjectURL(a.href);
  });

  // Drawer
  const drawer=document.getElementById('drawer'), dName=document.getElementById('dName'), dCode=document.getElementById('dCode'), dTeam=document.getElementById('dTeam'), dStatus=document.getElementById('dStatus'), dOpen=document.getElementById('dOpen'), dSub=document.getElementById('drawerSub'), dProgBar=document.getElementById('dProgBar'), dProgPct=document.getElementById('dProgPct');
  function openDrawerEl(el){
    dName.textContent = el.dataset.displayName || el.querySelector('.pt')?.textContent || '—';
    dCode.textContent = el.dataset.displayCode || el.querySelector('.pm b')?.textContent || '—';
    dTeam.textContent = el.dataset.displayTeam || el.querySelector('.pm')?.textContent?.split('Team: ')?.[1] || '—';
    const bucket = el.dataset.bucket || '';
    const stText = el.dataset.displayStatus || el.querySelector('.st')?.textContent || '—';
    dStatus.textContent = stText; dStatus.className='st '+bucket;
    const pct = parseInt(el.dataset.progress||'0',10); dProgBar.style.setProperty('--pct', pct+'%'); dProgPct.textContent=pct+'%';
    dOpen.href = el.getAttribute('href') || '#';
    dSub.textContent = bucket.replace('_',' ') + (dTeam.textContent? ' • '+dTeam.textContent : '');
    drawer.classList.add('open'); drawer.setAttribute('aria-hidden','false');
  }
  function closeDrawer(){ drawer.classList.remove('open'); drawer.setAttribute('aria-hidden','true'); }
  document.getElementById('drawerClose').onclick=closeDrawer; document.getElementById('drawerClose2').onclick=closeDrawer; drawer.addEventListener('click',(e)=>{ if(e.target===drawer) closeDrawer(); });

  function bindCardEvents(){
    $$('#board .project [data-open-detail]').forEach(btn=>{
      btn.addEventListener('click', (e)=>{ e.preventDefault(); e.stopPropagation(); openDrawerEl(btn.closest('.project')); });
    });
    $$('#board .project .pin').forEach(b=>{
      b.addEventListener('click', (e)=>{
        e.preventDefault(); e.stopPropagation();
        const card=b.closest('.project'); const id=card?.dataset.id; const cur=card.dataset.pinned==='true'; const nxt=!cur; card.dataset.pinned=String(nxt); b.textContent=nxt?'★':'☆';
        const obj=payloads.find(x=>x.id===id); if(obj) obj.pinned=nxt; persistPins(); if(sortSel.value==='pinned_first') renderBoard();
      });
    });
    $$('#projectsTableBody [data-open-detail]').forEach(btn=>{ btn.addEventListener('click', ()=> openDrawerEl(btn.closest('tr')) ); });
    $$('#projectsTableBody .pin').forEach(b=>{
      b.addEventListener('click', ()=>{
        const tr=b.closest('tr'); const id=tr?.dataset.id; const cur=tr.dataset.pinned==='true'; const nxt=!cur; tr.dataset.pinned=String(nxt); b.textContent=nxt?'★':'☆';
        const obj=payloads.find(x=>x.id===id); if(obj) obj.pinned=nxt; persistPins(); if(sortSel.value==='pinned_first') renderTable();
      });
    });
  }

  function persistPins(){ const arr=payloads.filter(p=>p.pinned).map(p=>p.id); LS.set('ppt_pins', arr); }
  function loadPins(){ const arr=LS.get('ppt_pins', []); if(Array.isArray(arr)){ payloads.forEach(p=> p.pinned = arr.includes(p.id)); } }

  // Restore state
  (function restore(){
    document.getElementById('q').value = LS.get('ppt_q','');
    document.getElementById('teamFilter').value = LS.get('ppt_team','');
    document.getElementById('statusFilter').value = LS.get('ppt_status','all');
    document.getElementById('sorter').value = LS.get('ppt_sort','name_asc');
    document.getElementById('board').setAttribute('data-density', LS.get('ppt_density','normal'));
    const view = LS.get('ppt_view','board');
    loadPins();
    if(view==='table'){ document.querySelectorAll('.seg-btn')[1].click(); } else { renderBoard(); }
    bindCardEvents();
  })();

  // Gauge fill animation + stroke length logic
  (function(){
    const score = parseInt(document.querySelector('.gauge').getAttribute('data-score')||'0',10);
    const path = document.getElementById('gFill');
    const length = path.getTotalLength ? path.getTotalLength() : 565; // fallback
    path.style.strokeDasharray = length;
    path.style.strokeDashoffset = length;
    requestAnimationFrame(()=>{
      const target = length * (1 - Math.min(100,Math.max(0,score))/100);
      path.style.strokeDashoffset = target;
    });
  })();

  // Keyboard: ESC to close drawer
  document.addEventListener('keydown', (e)=>{ if(e.key==='Escape') closeDrawer(); });
})();
</script>

<style>
:root{
  /* Forced white theme */
  --bg:#ffffff; --bg-soft:#f6f7fb; --card:#ffffff; --ink:#0f172a; --muted:#6b7280; --line:#e5e7eb;
  --primary:#0ea5e9; --primary-ink:#ffffff; --ok:#16a34a; --info:#2563eb; --warn:#f59e0b; --hold:#6b7280; --vio:#8b5cf6;
}
html,body{ height:100%; }
body{ margin:0; background:var(--bg); color:var(--ink); font:14px/1.55 system-ui,-apple-system,Segoe UI,Roboto,"Noto Sans Thai",sans-serif; }

.shell{ min-height:100vh; display:grid; grid-template-columns: 280px 1fr; }
@media (max-width: 960px){ .shell{ grid-template-columns: 1fr; } .side{ position:relative; order:2; } .main{ order:1; } }

/* Sidebar */
.side{ position:sticky; top:0; align-self:start; height:100vh; overflow:auto; background:linear-gradient(180deg,var(--card),var(--bg-soft)); border-right:1px solid var(--line); padding:16px; display:flex; flex-direction:column; gap:12px; }
.brand{ display:flex; gap:10px; align-items:center; }
.logo{ width:36px; height:36px; display:grid; place-items:center; border-radius:10px; background:var(--bg-soft); border:1px solid var(--line); }
.btx small{ color:var(--muted); display:block; margin-top:2px; }
.box{ background:var(--card); border:1px solid var(--line); border-radius:12px; padding:12px; display:flex; flex-direction:column; gap:10px; }
.box-h{ font-weight:800; font-size:13px; }
.lbl{ display:flex; flex-direction:column; gap:6px; font-size:12.5px; }
.lbl select{ padding:10px 12px; border:1px solid var(--line); border-radius:10px; background:var(--card); color:var(--ink); }
.search{ position:relative; }
.search input{ width:100%; padding:10px 32px 10px 12px; border:1px solid var(--line); border-radius:10px; background:var(--card); color:var(--ink); }
.search .btn.tiny{ position:absolute; right:4px; top:4px; height:28px; width:28px; border-radius:8px; border:1px solid var(--line); background:transparent; }
.grp{ display:grid; grid-template-columns:1fr 1fr; gap:8px; }
.btn{ padding:10px 12px; border-radius:10px; border:1px solid var(--line); background:var(--card); color:var(--ink); cursor:pointer; }
.btn.primary{ background:var(--primary); color:var(--primary-ink); border-color:transparent; font-weight:800; }
.btn.ghost{ background:transparent; }
.btn.w100{ width:100%; }
.foot .tips{ font-size:12px; color:var(--muted); }

/* Main */
.main{ display:flex; flex-direction:column; gap:12px; padding:16px; }
.top{ position:sticky; top:0; z-index:10; display:flex; justify-content:space-between; align-items:flex-start; gap:12px; padding:12px; background:rgba(255,255,255,.88); border:1px solid var(--line); border-radius:12px; backdrop-filter:saturate(120%) blur(10px); }
.top h1{ margin:0; font-size:clamp(18px,2vw,22px); font-weight:900; }
.top .sub{ margin:4px 0 0; color:var(--muted); font-size:12.5px; }
.actions{ display:flex; gap:8px; align-items:center; flex-wrap:wrap; }
.seg{ display:flex; background:var(--bg-soft); border:1px solid var(--line); border-radius:12px; overflow:hidden; }
.seg-btn{ padding:10px 14px; border:0; background:transparent; color:var(--muted); cursor:pointer; font-weight:700; }
.seg-btn.active{ background:var(--primary); color:var(--primary-ink); }

/* KPIs */
.kpis{ display:grid; gap:12px; grid-template-columns:repeat(6,minmax(0,1fr)); }
@media (max-width:1280px){ .kpis{ grid-template-columns:repeat(3,minmax(0,1fr)); } }
@media (max-width:640px){ .kpis{ grid-template-columns:repeat(2,minmax(0,1fr)); } }
.kpi{ background:var(--card); border:1px solid var(--line); border-radius:12px; padding:14px; }
.kpi .k{ color:var(--muted); font-size:12px; }
.kpi .v{ font-size:clamp(20px,3.2vw,28px); font-weight:900; margin-top:4px; }
.kpi.ok{ border-left:4px solid var(--ok); } .kpi.info{ border-left:4px solid var(--info); } .kpi.warn{ border-left:4px solid var(--warn); } .kpi.hold{ border-left:4px solid var(--hold); } .kpi.vio{ border-left:4px solid var(--vio); }

/* Charts */
.grid2{ display:grid; grid-template-columns:1fr 1fr; gap:12px; }
@media (max-width:1024px){ .grid2{ grid-template-columns:1fr; } }
.card{ background:var(--card); border:1px solid var(--line); border-radius:12px; padding:12px; }
.card .h{ font-weight:900; margin-bottom:10px; }
.donut{ position:relative; display:flex; align-items:center; justify-content:center; min-height:260px; }
.dsvg{ width:100%; max-width:360px; aspect-ratio:1/1; }
.ring{ transition: stroke-dashoffset .6s ease; }
.ring.completed{ stroke:var(--ok); } .ring.in_progress{ stroke:var(--info); }
.ring.planned{ stroke:var(--warn); } .ring.on_hold{ stroke:var(--hold); } .ring.other{ stroke:var(--vio); }
.donut-hole{ fill:var(--card); filter: drop-shadow(0 1px 0 rgba(0,0,0,.02)); }
.ctop{ font-size:12px; fill:var(--muted); font-weight:700; }
.cbot{ font-size:20px; font-weight:900; fill:var(--ink); }
.legend{ display:grid; grid-template-columns:repeat(auto-fit,minmax(180px,1fr)); gap:6px 12px; margin-top:10px; color:var(--muted); font-size:12.5px; }
.lgd{ display:flex; align-items:center; gap:8px; }
.lgd .dot{ width:10px; height:10px; border-radius:999px; background:var(--line); }
.lgd .dot.completed{ background:var(--ok); } .lgd .dot.in_progress{ background:var(--info); } .lgd .dot.planned{ background:var(--warn); } .lgd .dot.on_hold{ background:var(--hold); } .lgd .dot.other{ background:var(--vio); }
.tip{ position:absolute; pointer-events:none; background:#111827; color:#fff; font-size:12px; padding:6px 8px; border-radius:8px; transform:translate(-50%,-120%); white-space:nowrap; box-shadow:0 6px 18px rgba(0,0,0,.15); }
.tip.hidden{ display:none; }

/* Health gauge (semi-circle SVG) */
.health{ display:flex; flex-direction:column; gap:10px; }
.gauge{ position:relative; display:flex; align-items:center; justify-content:center; height:240px; border-radius:12px; background:linear-gradient(180deg,#fff,#f7fafc); }
.gsvg{ width:100%; max-width:420px; height:auto; }
.trk{ fill:none; stroke:var(--line); stroke-width:14; stroke-linecap:round; }
.fill{ fill:none; stroke:url(#gradGauge); stroke-width:14; stroke-linecap:round; stroke-dasharray:565; stroke-dashoffset:565; transition: stroke-dashoffset 1s ease; }
.tick{ stroke:#d1d5db; stroke-width:2; }
.gauge .n{ position:absolute; font-size:clamp(28px,4vw,36px); font-weight:900; }
.gauge .n span{ font-size:14px; color:var(--muted); margin-left:2px; }
.gauge .s{ position:absolute; bottom:10px; font-size:12px; color:var(--muted); }
.health .meta{ display:flex; gap:8px; flex-wrap:wrap; }
.pill{ padding:6px 10px; border-radius:999px; font-size:12px; border:1px solid var(--line); background:var(--card); color:var(--ink); }
.pill.ok{ border-color:var(--ok); } .pill.warn{ border-color:var(--warn); } .pill.crit{ border-color:#ef4444; }

/* Board */
.board{ display:grid; grid-template-columns: repeat(5, minmax(0,1fr)); gap:12px; }
@media (max-width:1400px){ .board{ grid-template-columns: repeat(3, minmax(0,1fr)); } }
@media (max-width:960px){ .board{ grid-template-columns: repeat(2, minmax(0,1fr)); } }
@media (max-width:640px){ .board{ grid-template-columns: 1fr; } }
.board[data-density="compact"] .project{ padding:10px; }
.col{ background:var(--card); border:1px solid var(--line); border-radius:12px; display:flex; flex-direction:column; min-height:200px; }
.col-h{ display:flex; align-items:center; gap:8px; padding:10px 12px; border-bottom:1px solid var(--line); position:sticky; top:0; background:var(--card); border-top-left-radius:12px; border-top-right-radius:12px; }
.badge{ width:10px; height:10px; border-radius:999px; background:var(--line); }
.badge.completed{ background:#a7f3d0; } .badge.in_progress{ background:#bfdbfe; } .badge.planned{ background:#fde68a; } .badge.on_hold{ background:#d1d5db; } .badge.other{ background:#ddd6fe; }
.col-body{ padding:10px; display:flex; flex-direction:column; gap:10px; }

.card.project{ text-decoration:none; color:inherit; border:1px solid var(--line); border-radius:12px; padding:12px; background:var(--card); transition:box-shadow .15s ease, transform .08s ease; display:flex; flex-direction:column; }
.card.project:hover{ box-shadow:0 10px 24px rgba(0,0,0,.10); transform:translateY(-1px); }
.ph{ display:flex; align-items:center; justify-content:space-between; gap:8px; }
.pt{ font-weight:900; font-size:14px; }
.pm{ margin-top:6px; color:var(--muted); font-size:12.5px; display:flex; gap:12px; flex-wrap:wrap; }
.pm2{ margin-top:6px; display:flex; align-items:center; justify-content:space-between; font-size:12px; color:var(--muted); }
.st{ padding:6px 10px; border-radius:999px; font-size:12px; border:1px solid transparent; white-space:nowrap; }
.st.completed{ background:#ecfdf5; color:#065f46; border-color:#a7f3d0; }
.st.in_progress{ background:#eff6ff; color:#1e40af; border-color:#bfdbfe; }
.st.planned{ background:#fffbeb; color:#92400e; border-color:#fde68a; }
.st.on_hold{ background:#f3f4f6; color:#374151; border-color:#e5e7eb; }
.st.other{ background:#f5f3ff; color:#5b21b6; border-color:#e9d5ff; }
.prog{ height:8px; border-radius:999px; background:var(--line); overflow:hidden; margin-top:8px; }
.prog.tiny{ height:6px; margin:0; }
.prog .bar{ height:100%; width:var(--pct); background:linear-gradient(90deg, var(--primary), #22d3ee); transition: width .25s ease; }
.pct{ font-weight:800; color:var(--ink); }
.pin{ border:1px solid var(--line); background:var(--card); border-radius:8px; padding:2px 8px; cursor:pointer; }
.pin:hover{ background:var(--bg-soft); }
.pf{ margin-top:6px; }
.link{ background:transparent; border:0; color:var(--primary); cursor:pointer; font-weight:700; }

/* Table */
.tablewrap{ background:var(--card); border:1px solid var(--line); border-radius:12px; overflow:auto; }
.tbl{ width:100%; border-collapse:separate; border-spacing:0; min-width:820px; }
.tbl th,.tbl td{ text-align:left; padding:10px 12px; border-bottom:1px solid var(--line); vertical-align:middle; }
.tbl thead th{ position:sticky; top:0; background:var(--bg-soft); z-index:1; }
.tbl tbody tr:hover{ background:#f9fafb; }
.tbl .t-name{ font-weight:700; }
.t-pct{ white-space:nowrap; display:flex; align-items:center; gap:8px; }
.pin-cell{ text-align:center; }

/* Empty */
.empty{ display:flex; flex-direction:column; align-items:center; justify-content:center; gap:6px; padding:40px 12px; color:var(--muted); }
.empty .emoji{ font-size:28px; } .empty .t1{ font-weight:900; }

/* Drawer */
.drawer{ position:fixed; inset:0; background:rgba(0,0,0,.35); display:none; align-items:stretch; justify-content:flex-end; }
.drawer.open{ display:flex; }
.panel{ width:min(520px,92vw); background:var(--card); border-left:1px solid var(--line); display:flex; flex-direction:column; }
.dh{ display:flex; align-items:flex-start; justify-content:space-between; gap:12px; padding:16px; border-bottom:1px solid var(--line); }
.ttl{ font-weight:900; font-size:16px; } .sub{ color:var(--muted); font-size:12.5px; margin-top:2px; }
.db{ padding:16px; display:flex; flex-direction:column; gap:10px; }
.kv{ display:flex; align-items:center; justify-content:space-between; gap:10px; padding:10px 0; border-bottom:1px dashed var(--line); }
.div{ height:1px; background:var(--line); margin:10px 0; }
.btitle{ font-weight:900; margin:6px 0; }
.bullets{ margin:0; padding-left:18px; }
.df{ padding:14px 16px; border-top:1px solid var(--line); display:flex; gap:8px; justify-content:flex-end; }

/* Motion */
@media (prefers-reduced-motion: reduce){ *{ transition:none !important; } }

/* SVG gradient for gauge */
.gauge{ --gstart:#ef4444; --gmid:#f59e0b; --gend:#16a34a; }
.gauge .gsvg{ position:relative; }
.gauge .gsvg::before{ content:""; position:absolute; inset:0; }
</style>

<!-- Hidden SVG defs for gauge gradient (once per page) -->
<svg width="0" height="0" style="position:absolute">
  <defs>
    <linearGradient id="gradGauge" x1="0%" y1="0%" x2="100%" y2="0%">
      <stop offset="0%" stop-color="#ef4444"/>
      <stop offset="50%" stop-color="#f59e0b"/>
      <stop offset="100%" stop-color="#16a34a"/>
    </linearGradient>
  </defs>
</svg>
